// Copyright (C) 2025 EDF
// All Rights Reserved
// This code is published under the GNU Lesser General Public License (GNU LGPL)
#ifndef TRANSITIONSTEPREGRESSIONDPCUTGRIDADAPT_H
#define TRANSITIONSTEPREGRESSIONDPCUTGRIDADAPT_H
#ifdef OMP
#include <omp.h>
#endif
#ifdef USE_MPI
#include <boost/mpi.hpp>
#endif
#include <memory>
#include <Eigen/Dense>
#include "geners/BinaryFileArchive.hh"
#include "StOpt/core/grids/GridAdaptBase.h"
#include "StOpt/dp/OptimizerDPCutGridAdaptBase.h"

/** \file TransitionStepRegressionDPCutGridAdapt.h
 * \brief Solve the dynamic programming  problem on one time step by regression with multi thread and mpi without distribution of the data
 *   The transition problem is written with cuts  so that the transition problem is written with  LP solver.
 * \author Xavier Warin
  */


namespace StOpt
{

/// \class TransitionStepRegressionDPCutGridAdapt TransitionStepRegressionDPCutGridAdapt.h
///        One step of dynamic programming without using mpi
class TransitionStepRegressionDPCutGridAdapt
{
private :


    std::shared_ptr<GridAdaptBase>  m_pGridCurrent ; ///< global grid at current time step
    std::shared_ptr<GridAdaptBase>  m_pGridPrevious ; ///< global grid at previous time step
     std::shared_ptr<OptimizerDPCutGridAdaptBase  >  m_pOptimize ; ///< optimizer solving the problem for one point and one step
#ifdef USE_MPI
    boost::mpi::communicator  m_world; ///< Mpi communicator
#endif

public :

    /// \brief Constructor
    TransitionStepRegressionDPCutGridAdapt(std::shared_ptr<GridAdaptBase> &p_pGridCurrent,
					     const  std::shared_ptr<GridAdaptBase> &p_pridPrevious,
					     const  std::shared_ptr<OptimizerDPCutGridAdaptBase > &p_pOptimize
#ifdef USE_MPI
					     , const boost::mpi::communicator &p_world
#endif
					     );

    /// \brief One step for dynamic programming in optimization
    /// \param p_phiIn      the function cut value ( (nb simulation * nb cuts), nb stocks ) coming from next step
    /// \param p_condExp    Conditional expectation object
    /// \return     vector containing the cut value for each ((simulation* nbcuts) * stock number)
    ///             each Eigen array has shape  (simulation* nbcuts) by  stock number
    Eigen::ArrayXXd   oneStep(const Eigen::ArrayXXd   &p_phiIn,	const std::shared_ptr< BaseRegression>     &p_condExp)  ;


    /// \brief Permits to dump continuation values on archive
    /// \param p_ar                   archive to dump in
    /// \param p_name                 name used for object
    /// \param p_iStep                Step number or identifier for time step
    /// \param p_phiIn                the function value ( nb simulation* nb cuts ,nb stocks)
    /// \param p_condExp              conditional expectation operator
    void dumpContinuationCutsValues(std::shared_ptr<gs::BinaryFileArchive> p_ar, const std::string &p_name, const int &p_iStep, const Eigen::ArrayXXd  &p_phiIn, const  std::shared_ptr<BaseRegression>    &p_condExp) const;

    /// \brief Permits to dump Bellmna values on archive
    /// \param p_ar                   archive to dump in
    /// \param p_name                 name used for object
    /// \param p_iStep                Step number or identifier for time step
    /// \param p_phiIn                the function value ( nb simulation* nb cuts ,nb stocks)
    /// \param p_condExp              conditional expectation operator
    void dumpBellmanCutsValues(std::shared_ptr<gs::BinaryFileArchive> p_ar, const std::string &p_name, const int &p_iStep, const Eigen::ArrayXXd  &p_phiIn, const  std::shared_ptr<BaseRegression>    &p_condExp) const;

};
}
#endif /* TRANSITIONSTEPREGRESSIONDPCUTGRIDADAPT_H */

